---
# R Code for: "Experimental vulture exclosure halves large carcass decomposition rates and doubles fly abundance"
# author: "Julia Koba Grootaers"
# date: "2025-02-26"
# output: html_document
---
  
##### Load Libraries ####

library(lmerTest)
library(jtools)
library(ggplot2)
library(gridExtra)
library(MuMIn)
library(glmmTMB)
library(plotrix)
library(dplyr)
library(tidyr)
library(Matrix)
library(lme4)


#### DECOMPOSITION RATE ANALYSIS ####

### Decomp Rate (weight lost/kg) ##
# The rate column seen in the dataset "Decomposition Rate Data_JG" which we will be using for our analysis was calculated by subsetting the data for each individual pig (e.g.: AGE001) for the first 10 days or until the carcass was consumed and then obtaining the slope from its linear regression: lm1<- lm(weight~ day, data=AGE001) from the "Raw Decomposition Data_JG" dataset. 
# The subset of each carcass weight data was created as follows once the first 10 days of data from each pig carcass were filtered from the dataset, this was done by hand to double check the data made sense.

#wet
AGE001<- subset(Decomp, pig_ID=="AGE001")
AGN001<- subset(Decomp, pig_ID=="AGN001")
DFE001<- subset(Decomp, pig_ID=="DFE001")
DFN001<- subset(Decomp, pig_ID=="DFN001")
PFN001<- subset(Decomp, pig_ID=="PFN001")
RFE001<- subset(Decomp, pig_ID=="RFE001")
RGE001<- subset(Decomp, pig_ID=="RGE001")
RGN001<- subset(Decomp, pig_ID=="RGN001")
AFN001<- subset(Decomp, pig_ID=="AFN001")
AGE002<- subset(Decomp, pig_ID=="AGE002")
AGN002<- subset(Decomp, pig_ID=="AGN002")
TFE001<- subset(Decomp, pig_ID=="TFE001")
AGN003<- subset(Decomp, pig_ID=="AGN003")
CFE001<- subset(Decomp, pig_ID=="CFE001")
FFN001<- subset(Decomp, pig_ID=="FFN001")
RGE002<- subset(Decomp, pig_ID=="RGE002")


#dry
RGN002<- subset(Decomp, pig_ID=="RGN002")
RGE003<- subset(Decomp, pig_ID=="RGE003")
DFN002<- subset(Decomp, pig_ID=="DFN002")
DFE002<- subset(Decomp, pig_ID=="DFE002")
RGE004<- subset(Decomp, pig_ID=="RGE004")
CFN002<- subset(Decomp, pig_ID=="CFN002")
FGN001<- subset(Decomp, pig_ID=="FGN001")
TFE002<- subset(Decomp, pig_ID=="TFE002")
CFN003<- subset(Decomp, pig_ID=="CFN003")
AFE001<- subset(Decomp, pig_ID=="AFE001")
RGE005<- subset(Decomp, pig_ID=="RGE005")
RGN003<- subset(Decomp, pig_ID=="RGN003")
AGE003<- subset(Decomp, pig_ID=="AGE003")
FGN002<- subset(Decomp, pig_ID=="FGN002")
CFE002<- subset(Decomp, pig_ID=="CFE002")
CFN004<- subset(Decomp, pig_ID=="CFN004")


# LM to every carcass to obtain slope

summary(lm13)
#wet  

lm1<- lm(weight ~ day, data=AGE001) 
lm2<- lm(weight ~ day, data=AGN001) 
lm3<- lm(weight ~ day, data=DFE001) 
lm4<- lm(weight ~ day, data=DFN001) 
lm5<- lm(weight ~ day, data=PFN001) 
lm6<- lm(weight ~ day, data=RFE001) 
lm7<- lm(weight ~ day, data=RGE001) 
lm8<- lm(weight ~ day, data=RGN001) 
lm9<- lm(weight ~ day, data=AFN001) 
lm10<- lm(weight ~ day, data=AGE002) 
lm11<- lm(weight ~ day, data=AGN002) 
lm12<- lm(weight ~ day, data=TFE001) 
lm13<- lm(weight ~ day, data=AGN003) 
lm14<- lm(weight ~ day, data=CFE001) 
lm15<- lm(weight ~ day, data=FFN001) 
lm16<- lm(weight ~ day, data=RGE002) 


#dry
lm17<- lm(weight ~ day, data=RGN002) 
lm18<- lm(weight ~ day, data=RGE003) 
lm19<- lm(weight ~ day, data=DFN002) 
lm20<- lm(weight ~ day, data=DFE002) 
lm21<- lm(weight ~ day, data=RGE004) 
lm22<- lm(weight ~ day, data=CFN002) 
lm23<- lm(weight ~ day, data=FGN001) 
lm24<- lm(weight ~ day, data=TFE002) 
lm25<- lm(weight ~ day, data=CFN003) 
lm26<- lm(weight ~ day, data=AFE001) 
lm27<- lm(weight ~ day, data=RGE005) 
lm28<- lm(weight ~ day, data=RGN003) 
lm29<- lm(weight ~ day, data=AGE003) 
lm30<- lm(weight ~ day, data=FGN002) 
lm31<- lm(weight ~ day, data=CFE002) 
lm32<- lm(weight ~ day, data=CFN004) 

summary(lm3)

#### For ease, we will use the dataset "Decomposition Rate Data_JG" which already has the rate column in it.
## Load data.

Decomp <- read.table("Decomposition Rate Data_JG.txt", header = TRUE)
View(Decomp)


## Explore data.

table(Decomp$pig_ID) # n= 32 pigs
summary (Decomp)
str(Decomp) # Treatment, round and habitat need to be converted to factors. 


## Convert to factors 

Decomp$treatment <- as.factor(Decomp$treatment)
summary(Decomp$treatment) # 2 treatments: exclosure and control. 16 observations each.
Decomp$round <- as.factor(Decomp$round)
summary(Decomp$round) # 8 rounds. 4 observations in each.
Decomp$habitat <- as.factor(Decomp$habitat)
summary(Decomp$habitat) # two habitats: grassland and forest. 16 observations each.
Decomp$season <- as.factor(Decomp$season)
summary(Decomp$season) # two seasons: wet and dry. 16 observations each.


## Check normality of response variable rate.

lm<- lm(rate ~ treatment, data= Decomp)
summary(lm)
par(mfrow=c(1,1))

plot (lm) # QQ plot 
hist(Decomp$rate) # Right skewed- will always be right skewed because of the nature of carcasses decomposing over time.
var(Decomp$rate)


## Plot decomposition rate in treatments (control and exclosure) and check outliers.

boxplot(rate ~ treatment*habitat , data= Decomp)$out 

## Compare full models with and without initial carcass mass ##

glmm_full1 <- glmmTMB(rate ~ treatment * habitat + season + (1|round), data=Decomp)
glmm_full2 <- glmmTMB(rate ~ treatment * season + habitat + (1|round), data=Decomp)

glmm_full1_weight <- glmmTMB(rate ~ treatment * habitat + season + initial_weight + (1|round), data=Decomp)
glmm_full2_weight <- glmmTMB(rate ~ treatment * season + habitat + initial_weight + (1|round), data=Decomp)

AIC(glmm_full1, glmm_full1_weight)#initial carcass mass does not improve model fit
AIC(glmm_full2, glmm_full2_weight) #initial carcass mass does not improve model fit

# Exclude initial carcass mass from model selection to reduce model complexity.

###### Table 1. Model selection  ######

glmm  <- glmmTMB(rate ~ treatment+(1|round), data=Decomp)
glmm1 <- glmmTMB(rate ~ treatment+habitat+(1|round), data=Decomp)
glmm2 <- glmmTMB(rate ~ treatment+habitat+season+(1|round), data=Decomp)
glmm3 <- glmmTMB(rate ~ treatment*habitat+(1|round), data=Decomp)
glmm4 <- glmmTMB(rate ~ treatment*habitat+season+(1|round), data=Decomp)
glmm5 <- glmmTMB(rate ~ treatment+season+(1|round), data=Decomp)
glmm6 <- glmmTMB(rate ~ treatment*season+(1|round), data=Decomp)
glmm7 <- glmmTMB(rate ~ treatment*season+habitat+(1|round), data=Decomp)
glmm8 <- glmmTMB(rate ~ habitat+(1|round), data=Decomp)
glmm9 <- glmmTMB(rate ~ season+(1|round), data=Decomp)
glmmNull <- glmmTMB(rate ~ (1|round), data=Decomp)

model_table<-model.sel(glmm,glmm1,glmm2,glmm3,glmm4,glmm5, glmm6, glmm7, glmm8, glmm9,glmmNull) #glmm9 is best fit, with just treatment.
print(model_table)
r.squaredGLMM(glmm)
summary(glmm6) # Treatment only model is the best fit
confint(glmm6)


###### Plot Decomposition Rate ####
## Use Jtools to plot decomposition rate in the treatments (control & exclosure) and habitats (forest & grassland).

# Control

lmcon<- lm(rate ~ habitat, data= subset(Decomp, treatment=="Control"))
plot1<- effect_plot(lmcon, pred = habitat, interval = TRUE,robust= TRUE, plot.points = TRUE,  jitter = .1, main.title= "Control", x.label= "Habitat", y.label= "Decomposition rate (kg lost/day)", line.thickness= 0.7) +
       theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0,20)

# Exclosure

lmexc<- lm(rate ~ habitat, data= subset(Decomp, treatment=="Exclosure"))
plot2<- effect_plot(lmexc, pred = habitat, interval = TRUE, robust= TRUE, plot.points = TRUE,
           jitter = .1, main.title= "Exclosure", x.label= "Habitat", y.label= "Decomposition rate (kg lost/day)",line.thickness= 0.7) +
 theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0,20)

## Plot both in one grid.

plotdecomp<-grid.arrange(plot1, plot2, nrow=1, ncol=2)

## Plot both in one graph 

plot2<- effect_plot(glmm, pred = treatment) 

plotCI(c(1.5,2.9),
       c(plot2$data$rate),
       ui=c(plot2$data$ymax),
       li=c(plot2$data$ymin),
       xlim=c(0.9,3.5),las=1,
       ylim=c(0,17),
       pch=19, sfrac=0.00001,
       ylab="Predicted decomposition rate (kg lost/day)",
       xlab="",
       xaxt="n", 
       lwd=2)
abline(v=2.2, lty=2)

## Add raw data

# Forest control
points( jitter(rep(1.2, time=8),1), Decomp$rate[Decomp$treatment=="Control" & Decomp$habitat=="Forest"], pch=19, cex=0.7,col=rgb(0,0,0,0.4))
# Grassland control
points( jitter(rep(1.8, time=8),1), Decomp$rate[Decomp$treatment=="Control" & Decomp$habitat=="Grassland"], pch=19, cex=0.7,col=rgb(0,0,0,0.4))
# Forest Exclosure
points( jitter(rep(2.6, time=8),1), Decomp$rate[Decomp$treatment=="Exclosure" & Decomp$habitat=="Forest"], pch=19, cex=0.7,col=rgb(0,0,0,0.4))
# Grassland Exclosure
points( jitter(rep(3.2, time=8),1), Decomp$rate[Decomp$treatment=="Exclosure" & Decomp$habitat=="Grassland"], pch=19, cex=0.7,col=rgb(0,0,0,0.4))

axis(1, at=c(1.2,1.8,2.6,3.2),
     labels=c("Forest", "Grassland", "Forest", "Grassland"))


mtext(expression(bold("Control")), las=1,
      side=1,
      line=2.4,
      outer = FALSE,
      at=c(1.5)
)

mtext(text=expression(bold("Exclosure")), las=1,
      side=1,
      line=2.4,
      outer = FALSE,
      at=c(2.9)
)



#### VULTURE ABUNDANCE ANALYSIS ####

###### Calculate vulture RAI #######

# Below is how we calculated vulture RAI, if you want you can skip to the analysis with the ready calculated RAI dataset please see section 'RAI Analysis'.


# The Relative Abundance Index (RAI) per pig was calculated as the number of individual vultures in all images during the first 10 days of a pig's persistence or until the pig was consumed (if consumed before the first 10 days) divided by the number of days the pig persisted (either 10 days or less). 
# Camera traps were set to take images every 60 seconds. There is one RAI per pig. 
# Exclosure camera trap footage is unreliable because vultures do not always descend to the ground as they can observe the carcass and exclosure from the trees. Because of this and the fact that some cameras malfunctioned at exclosure sites, I have excluded the exclosure data from the analysis.

## Load libraries
library(dplyr)
library(lubridate)
library(ggplot2)
library(reshape2)
library(tidyverse)

## Import data sets 

# Wildlife insights metadata (identifications and counts)

metadata <- read.csv("C:/Users/Raw Metadata_JG.csv", stringsAsFactors=TRUE)

# Deployments info (start and end date)

minfo <- read.csv("C:/Users/Raw Deployment_data_JG.csv", stringsAsFactors=TRUE)

# End date is the day when the carcass was completely consumed (if < 10 days after day 0), or else: date of day no.10

## Clean, prepare and merge datasets
# clean minfo 

minfo <- minfo[minfo$Usable!= "n", ]

# merge image and info datasets

data <- merge(metadata, minfo, by = "deployment_id", all.x = F) 


data= subset(data, data$order== "Cathartiformes")

# check everything is there

summary(data$deployment_id) # 29, as many as in metadata dataset ?

levels(metadata$deployment_id) # 29

# Ensure timestamps are in POSIXct format
data$timestamp <- as.POSIXct(data$timestamp, format = "%d.%m.%Y %H:%M")
data$End_date <- as.POSIXct(data$End_date, format = "%d.%m.%Y")
data$Start_date <- as.POSIXct(data$Start_date, format = "%d.%m.%Y")

data$timestamp <- as_datetime(data$timestamp) 
data$Start_date <- as_date(data$Start_date) 
data$End_date <- as_date(data$End_date) 


# Combine Start_date and time_deployed into a single character string
data$datetime <- paste(data$Start_date, data$Time_deployed)

# Convert the combined character string into POSIXct datetime
data$datetime <- as.POSIXct(data$datetime, format = "%Y-%m-%d %H:%M")

View(data)
# Convert Start_date to start of the day

# Convert End_date to end of the day
data$End_date <- as.Date(data$End_date) + days(1) - seconds(1) # add a day to end date and then minus 1 second so that the full day in end date is used (eg. end of the day is 13.05.2024 23:59).


# Filter data to include only observations within the range
clean_data <- data %>%
  filter(timestamp >= datetime & timestamp <= End_date) # filter observations that are between the start and end date (including the end date)


## Select first picture of every minute for deployments where picture delay was set to 1 s

# List of specific deployment IDs to process
target_deployment_ids <- c("RGE004_C508_2024-03-06", "CFN002_C509_2024-03-06", "TFE002_C511_2024-03-06")

# Process only rows with the specified deployment IDs
processed <- clean_data %>%
  filter(deployment_id %in% target_deployment_ids) %>%
  group_by(deployment_id, common_name, timestamp) %>%
  slice(1) %>%
  ungroup() 

# Combine the processed rows with the rest of the original data
new_data <- bind_rows(
  processed,
  clean_data %>%
    filter(!deployment_id %in% target_deployment_ids))

summary(new_data$deployment_id)

## RAI calculation ##

# calculate persistence 
new_data$persistence <- as.numeric(difftime(new_data$End_date, new_data$datetime, units = "days")) 

summary(new_data$deployment_id)

# Calculate RAI
RAI <- new_data %>%
  group_by(deployment_id) %>%
  summarise(RAI = sum(number_of_objects) / first(persistence))

summary(RAI$deployment_id)

final_data <- merge(new_data, RAI, by = "deployment_id", all.x = TRUE)

### Save new dataframe with one RAI for each deployment ID ###

Vult.RAI <- final_data %>%
  group_by(deployment_id) %>%
  slice(1) %>%
  ungroup()

summary(Vult.RAI$deployment_id) 
View(Vult.RAI)

write.csv(Vult.RAI, file= "C:/Users/Vulture Abundance.csv", row.names=FALSE)
# In excel, add in FGN001, no vultures arrived so it was not uploaded to wildlife insights, RAI is 0.


###### RAI Analysis ########

## To do the analysis, we have prepared the dataset "Vulture Abundance Data_JG" which contains the full RAI column.

## Load data.
Vult <- read.table("Vulture Abundance Data_JG.txt", header= TRUE)
View(Vult)


## Explore Data.

table(Vult$pig_ID) 
counts.id<- table(Vult$pig_ID)
table(counts.id) # n= 26 pigs (the other 6 cameras malfunctioned)
summary (Vult)
str(Vult) # Treatment, round and habitat need to be converted to factors. 

## Convert to factors

Vult$treatment <- as.factor(Vult$treatment)
summary(Vult$treatment)
Vult$habitat <- as.factor(Vult$habitat)
summary(Vult$habitat)  
Vult$season <- as.factor(Vult$season)
summary(Vult$season)
Vult$round <- as.factor(Vult$round)
summary(Vult$round)
Vult$RAI <- as.numeric(Vult$RAI)
summary(Vult$RAI)

## Take out exclosures.

Con_RAI<- subset(Vult, treatment =="Control")
str(Con_RAI)
summary(Con_RAI) # season: 8 dry and 8 wet; habitat: 8 forest and 8 grassland.
View(Con_RAI)


## Check normality of residuals - response variable RAI

lm<- lm(RAI , data= Con_RAI)
summary(lm)
plot (lm) 
hist(Con_RAI$RAI) # right skewed
var(RAI)
View(Con_RAI)

## Check outliers.

boxplot(RAI ~ habitat, data= Con_RAI)$out #none

###### Effect of vulture abundance on decomposition rate #####

lm1 <- lm(weight_rate ~ RAI, data=Con_RAI)
lm2 <- lm(weight_rate ~ 1 , data=Con_RAI)
model.sel(lm1,lm2) # lm1 better fit, with RAI. 

summary(lm1)
confint(lm1)

## Plot decomposition rate in control sites on relative vulture abundance
plot(Con_RAI$RAI, Con_RAI$weight_rate)


###### Fig. 2. Decomposition rate and relative vulture abundance ######

## Plot Decomposition Rate by Season ##

# Wet Season - Control

lmcon_wet <- lm(rate ~ habitat, data = subset(Decomp, treatment == "Control" & season == "wet"))
plot1_wet <- effect_plot(
  lmcon_wet, pred = habitat, interval = TRUE, robust = TRUE, plot.points = TRUE, 
  jitter = 0.1, main.title = "Control - Wet Season", 
  x.label = "Habitat", y.label = "Decomposition rate (kg lost/day)", line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Dry Season - Control
lmcon_dry <- lm(rate ~ habitat, data = subset(Decomp, treatment == "Control" & season == "dry"))
plot1_dry <- effect_plot(
  lmcon_dry, pred = habitat, interval = TRUE, robust = TRUE, plot.points = TRUE, 
  jitter = 0.1, main.title = "Control - Dry Season", 
  x.label = "Habitat", y.label = "Decomposition rate (kg lost/day)", line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Wet Season - Exclosure
lmexc_wet <- lm(rate ~ habitat, data = subset(Decomp, treatment == "Exclosure" & season == "wet"))
plot2_wet <- effect_plot(
  lmexc_wet, pred = habitat, interval = TRUE, robust = TRUE, plot.points = TRUE, 
  jitter = 0.1, main.title = "Exclosure - Wet Season", 
  x.label = "Habitat", y.label = "Decomposition rate (kg lost/day)", line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Dry Season - Exclosure
lmexc_dry <- lm(rate ~ habitat, data = subset(Decomp, treatment == "Exclosure" & season == "dry"))
plot2_dry <- effect_plot(
  lmexc_dry, pred = habitat, interval = TRUE, robust = TRUE, plot.points = TRUE, 
  jitter = 0.1, main.title = "Exclosure - Dry Season", 
  x.label = "Habitat", y.label = "Decomposition rate (kg lost/day)", line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Combine Plots in a Grid
par(mfrow=c(1,1))
plot_decomp_season <- grid.arrange(plot1_wet, plot1_dry, plot2_wet, plot2_dry, nrow = 2, ncol = 2)

# Combined Graph for Both Treatments Across Seasons
plot_combined <- effect_plot(glmm, pred = treatment) 


### Panel A ###


pdf("Figure2.pdf", height=6, width=10)

par(mfrow=c(1,2))
plotCI(
  c(1.5, 2.9), c(plot_combined$data$rate),
  ui = c(plot_combined$data$ymax),
  li = c(plot_combined$data$ymin),
  xlim = c(0.9, 3.5), las = 1,
  ylim = c(0, 17), pch = 19, sfrac = 0.00001,
  ylab = expression("Carcass decomposition rate (kg lost day"^{-1}*")"),
  xlab = "",
  xaxt = "n", lwd = 2
)
abline(v = 2.2, lty = 2)

mtext("A)", side = 3, line = 1, adj = 0, font = 2, cex = 1.2, col = "grey30")


# Add Raw Data for Wet and Dry Seasons

# Forest control (Wet)
n_points <- sum(Decomp$treatment == "Control" & Decomp$habitat == "Forest" & Decomp$season == "wet")
points(jitter(rep(1.2, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Control" & Decomp$habitat == "Forest" & Decomp$season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Forest control (Dry)
n_points <- sum(Decomp$treatment == "Control" & Decomp$habitat == "Forest" & Decomp$season == "dry")
points(jitter(rep(1.2, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Control" & Decomp$habitat == "Forest" & Decomp$season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Grassland control (Wet)
n_points <- sum(Decomp$treatment == "Control" & Decomp$habitat == "Grassland" & Decomp$season == "wet")
points(jitter(rep(1.8, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Control" & Decomp$habitat == "Grassland" & Decomp$season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Grassland control (Dry)
n_points <- sum(Decomp$treatment == "Control" & Decomp$habitat == "Grassland" & Decomp$season == "dry")
points(jitter(rep(1.8, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Control" & Decomp$habitat == "Grassland" & Decomp$season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Forest exclosure (Wet)
n_points <- sum(Decomp$treatment == "Exclosure" & Decomp$habitat == "Forest" & Decomp$season == "wet")
points(jitter(rep(2.6, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Exclosure" & Decomp$habitat == "Forest" & Decomp$season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Forest exclosure (Dry)
n_points <- sum(Decomp$treatment == "Exclosure" & Decomp$habitat == "Forest" & Decomp$season == "dry")
points(jitter(rep(2.6, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Exclosure" & Decomp$habitat == "Forest" & Decomp$season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Grassland exclosure (Wet)
n_points <- sum(Decomp$treatment == "Exclosure" & Decomp$habitat == "Grassland" & Decomp$season == "wet")
points(jitter(rep(3.2, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Exclosure" & Decomp$habitat == "Grassland" & Decomp$season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Grassland exclosure (Dry)
n_points <- sum(Decomp$treatment == "Exclosure" & Decomp$habitat == "Grassland" & Decomp$season == "dry")
points(jitter(rep(3.2, time = n_points), 1), 
       Decomp$rate[Decomp$treatment == "Exclosure" & Decomp$habitat == "Grassland" & Decomp$season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

axis(1, at = c(1.2, 1.8, 2.6, 3.2),
     labels = c("Forest", "Grassland", "Forest", "Grassland"))

lines(c(1.5, 2.9),
      c(plot_combined$data$rate), lty=2)

mtext(expression(bold("Control")), las = 1, side = 1, line = 2.4, outer = FALSE, at = c(1.5))
mtext(expression(bold("Exclusion")), las = 1, side = 1, line = 2.4, outer = FALSE, at = c(2.9))

## Panel B

# Determine dynamic ylim upper bound
upper_ylim <- max(plot1$data$ymax) + 1  # Adding a buffer of 1 unit to improve presentation of the plot

plot1 <- effect_plot(lm1, pred = RAI, robust = TRUE, interval = TRUE, 
                     plot.points = TRUE, jitter = .1, 
                     main.title = "Control", 
                     x.label = "Habitat", 
                     y.label = "Decomposition rate (kg lost/day)", 
                     line.thickness = 0.7) +
  theme(plot.title.position = 'plot', 
        plot.title = element_text(hjust = 0.7)) +
  ylim(0, upper_ylim)  # Use dynamic ylim

# THE PREDICTIONS
plot1$data
first(plot1$data) # low RAI
last(plot1$data) # High RAI


plot(Con_RAI$weight_rate ~ Con_RAI$RAI, ylim = c(0, upper_ylim), las = 1,
     ylab = expression("Carcass decomposition rate (kg lost day"^{-1}*")"),
     xlab = "Relative vulture abundance", type = "n")

polygon(c(plot1$data$RAI, rev(plot1$data$RAI)),
        c(plot1$data$ymax, rev(plot1$data$ymin)),
        lwd = 2, border = rgb(0.9, 0.9, 0.9), 
        col = rgb(0.9, 0.9, 0.9))

lines(plot1$data$weight_rate ~ plot1$data$RAI, lwd = 2)

points(Con_RAI$weight_rate ~ Con_RAI$RAI, pch = 19,
       cex = 0.7, col = rgb(0, 0, 0, 0.4))
mtext("B)", side = 3, line = 1, adj = 0, font = 2, cex = 1.2, col = "grey30")

dev.off()




#### FLY ANALYSIS ####

## Load data.
# The abundance column seen in the data was calculated extracting the mean nr of flies per pig (total nr flies/days) using: mean(subset(Fly, pig_ID=="AFN001")$count) from the dataset "Raw Invertebrate Data_JG". Replacing the "AFN001" with the relevant pig_ID. 
# The results were then compiled into the new data set we will now load for the analysis.

Fly <- read.table("Fly Abundance Data_JG.txt", header = TRUE)
View(Fly)


## Explore data.

table(Fly$pig_ID) # n= 32 pigs
summary (Fly) # 32 pigs, 8 rounds.
str(Fly) # Treatment, round and habitat need to be converted to factors. 

## Convert to factors.
Fly$treatment <- as.factor(Fly$treatment)
summary(Fly$treatment) # 2 treatments: Control & Exclosure. 16 observations each.
Fly$habitat <- as.factor(Fly$habitat)
summary(Fly$habitat) # 2 habitats: Forest & Grassland. 16 observations each.
Fly$round <- as.factor(Fly$round)
summary(Fly$round) # 8 rounds. 4 observations each.
Fly$season <- as.factor(Fly$season)
summary(Fly$season) # 2 seasons: Wet & Dry. 16 observations each.


# Check normality of residuals - response variable abundance.

lm<- lm(abundance ~ treatment, data= Fly)
plot (lm) # QQ plot 
hist(Fly$abundance) # right skewed.

## Plot abundance in treatments (control and exclosure) and check outliers.

boxplot(abundance ~ treatment , data= Fly)$out #Outlier: pig RFE001 in round 2 was gone uncommonly quickly for a forest treatment and had many flies so the mean is much higher than usual.


###### Table 2. Model selection ######

Fly$abun_integer <- round(Fly$abundance,0)

lm  <-  glmmTMB(abun_integer ~ treatment + (1|round), data=Fly, family=nbinom2)
lm1 <-  glmmTMB(abun_integer ~ habitat + (1|round), data=Fly, family=nbinom2)
lm2 <-  glmmTMB(abun_integer ~ season + (1|round), data=Fly, family=nbinom2)
lm3 <-  glmmTMB(abun_integer ~ treatment + habitat + (1|round), data=Fly, family=nbinom2)
lm4 <-  glmmTMB(abun_integer ~ treatment + habitat + season + (1|round), data=Fly, family=nbinom2)
lm5 <-  glmmTMB(abun_integer ~ treatment*habitat + (1|round), data=Fly, family=nbinom2)
lm6 <-  glmmTMB(abun_integer ~ treatment*habitat + season + (1|round), data=Fly, family=nbinom2)
lm7 <-  glmmTMB(abun_integer ~ treatment + season + (1|round), data=Fly, family=nbinom2)
lm8 <-  glmmTMB(abun_integer ~ treatment*season + (1|round), data=Fly, family=nbinom2)
lm9 <-  glmmTMB(abun_integer ~ treatment*season + habitat + (1|round), data=Fly, family=nbinom2)
lmNull <-  glmmTMB(abun_integer ~ (1|round), data=Fly, family=nbinom2)

model_table <-model.sel(lm,lm1,lm2,lm3,lm4,lm5,lm6,lm7,lm8, lm9, lmNull) #lm is best fit model.
print(model_table)
summary(lm)
confint(lm)
r.squaredGLMM(lm)


## Does fly abundance affect difference in carcass decomposition in the absence of vultures?

Fly_ex<- subset(Fly, treatment=="Exclosure")
head(Decomp)
Fly_ex <- left_join(Fly_ex, Decomp[,c("pig_ID", "rate")])
View(Fly_ex)

summary(lm(Fly_ex$rate~Fly_ex$abun_integer))

rlm1 <- lm(rate ~ abun_integer, data=Fly_ex)
rlm2 <- lm(rate ~ 1 , data=Fly_ex)
model.sel(rlm1,rlm2) # rlm1 better fit, with RAI. 
summary(rlm1) # Fly abundance has a significant effect on decomposition rate at exclosure sites.

###### Fig. 3. Fly Abundance and Decomposition Rate ######

## Plot Fly Abundance by Season ##

# Wet Season - Control
lmcon_Wet <- lm(abun_integer ~ habitat, data = subset(Fly, treatment == "Control" & season == "Wet"))
plotF1_Wet <- effect_plot(
  lmcon_Wet, pred = habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1, 
  main.title = "Control - Wet Season", x.label = "Habitat", y.label = "Fly Abundance (nr/day)", 
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 220)

# Dry Season - Control
lmcon_Dry <- lm(abun_integer ~ habitat, data = subset(Fly, treatment == "Control" & season == "Dry"))
plotF1_Dry <- effect_plot(
  lmcon_Dry, pred = habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1, 
  main.title = "Control - Dry Season", x.label = "Habitat", y.label = "Fly Abundance (nr/day)", 
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 220)

# Wet Season - Exclosure
lmexc_Wet <- lm(abun_integer ~ habitat, data = subset(Fly, treatment == "Exclosure" & season == "Wet"))
plotF2_Wet <- effect_plot(
  lmexc_Wet, pred = habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1, 
  main.title = "Exclosure - Wet Season", x.label = "Habitat", y.label = "Fly Abundance (nr/day)", 
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 220)

# Dry Season - Exclosure
lmexc_Dry <- lm(abun_integer ~ habitat, data = subset(Fly, treatment == "Exclosure" & season == "Dry"))
plotF2_Dry <- effect_plot(
  lmexc_Dry, pred = habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1, 
  main.title = "Exclosure - Dry Season", x.label = "Habitat", y.label = "Fly Abundance (nr/day)", 
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 220)


# Combine Plots in a Grid
par(mfrow = c(1, 2))
plot_fly_season <- grid.arrange(plotF1_Wet, plotF1_Dry, plotF2_Wet, plotF2_Dry, nrow = 2, ncol = 2)

# Combined Graph for Both Treatments Across Seasons
plot_combined <- effect_plot(lm, pred = treatment) 



### Panel A ###

pdf("Figure3.pdf", height=6, width=10)

par(mfrow=c(1,2))
plotCI(
  c(1.5, 2.9), c(plot_combined$data$abun_integer),
  ui = c(plot_combined$data$ymax),
  li = c(plot_combined$data$ymin),
  xlim = c(0.9, 3.5), las = 1,
  ylim = c(0, 220), pch = 19, sfrac = 0.00001,
  ylab = expression("Fly abundance (nr day"^{-1}*")"),
  xlab = "", xaxt = "n", lwd = 2
)
abline(v = 2.2, lty = 2)
mtext("A)", side = 3, line = 1, adj = 0, font = 2, cex = 1.2, col = "grey30")

# Add Raw Data for Wet and Dry Seasons

# Forest control (Wet)
n_points <- sum(Fly$treatment == "Control" & Fly$habitat == "Forest" & Fly$season == "Wet")
points(jitter(rep(1.2, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Control" & Fly$habitat == "Forest" & Fly$season == "Wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet


# Forest control (Dry)
n_points <- sum(Fly$treatment == "Control" & Fly$habitat == "Forest" & Fly$season == "Dry")
points(jitter(rep(1.2, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Control" & Fly$habitat == "Forest" & Fly$season == "Dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Grassland control (Wet)
n_points <- sum(Fly$treatment == "Control" & Fly$habitat == "Grassland" & Fly$season == "Wet")
points(jitter(rep(1.8, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Control" & Fly$habitat == "Grassland" & Fly$season == "Wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Grassland control (Dry)
n_points <- sum(Fly$treatment == "Control" & Fly$habitat == "Grassland" & Fly$season == "Dry")
points(jitter(rep(1.8, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Control" & Fly$habitat == "Grassland" & Fly$season == "Dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Forest exclosure (Wet)
n_points <- sum(Fly$treatment == "Exclosure" & Fly$habitat == "Forest" & Fly$season == "Wet")
points(jitter(rep(2.6, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Exclosure" & Fly$habitat == "Forest" & Fly$season == "Wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Forest exclosure (Dry)
n_points <- sum(Fly$treatment == "Exclosure" & Fly$habitat == "Forest" & Fly$season == "Dry")
points(jitter(rep(2.6, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Exclosure" & Fly$habitat == "Forest" & Fly$season == "Dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Grassland exclosure (Wet)
n_points <- sum(Fly$treatment == "Exclosure" & Fly$habitat == "Grassland" & Fly$season == "Wet")
points(jitter(rep(3.2, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Exclosure" & Fly$habitat == "Grassland" & Fly$season == "Wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Grassland exclosure (Dry)
n_points <- sum(Fly$treatment == "Exclosure" & Fly$habitat == "Grassland" & Fly$season == "Dry")
points(jitter(rep(3.2, time = n_points), 1), 
       Fly$abun_integer[Fly$treatment == "Exclosure" & Fly$habitat == "Grassland" & Fly$season == "Dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

axis(1, at = c(1.2, 1.8, 2.6, 3.2),
     labels = c("Forest", "Grassland", "Forest", "Grassland"))

mtext(expression(bold("Control")), las = 1, side = 1, line = 2.4, outer = FALSE, at = c(1.5))
mtext(expression(bold("Exclusion")), las = 1, side = 1, line = 2.4, outer = FALSE, at = c(2.9))

lines(c(1.5, 2.9),
      c(plot_combined$data$abun_integer), lty=2)


#### Panel B ##

plot1<- effect_plot(rlm1, pred = abun_integer, interval = TRUE,robust= TRUE, plot.points = TRUE,  jitter = .1, main.title= "Control", x.label= "Habitat", y.label= "Decomposition rate (kg lost/day)", line.thickness= 0.7) +
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0,20)

# The Predictions
plot1$data
first(plot1$data) # low RAI
last(plot1$data) # High RAI

plot(Fly_ex$rate~Fly_ex$abun_integer, ylim=c(0,13),xlim=c(0,210), las=1,
     ylab=expression("Carcass decomposition rate (kg lost day"^{-1}*")"),
     xlab=expression("Fly abundance (nr day"^{-1}*")"), type="n")

polygon(c(plot1$data$abun_integer,rev(plot1$data$abun_integer)),
        c(plot1$data$ymax,rev(plot1$data$ymin)),
        lwd=2,border=rgb(0.9,0.9,0.9), col=rgb(0.9,0.9,0.9) )
lines(plot1$data$rate~plot1$data$abun_integer, lwd=2)
points(Fly_ex$rate~Fly_ex$abun_integer,pch=19,
       cex=0.7,col=rgb(0,0,0,0.4))

mtext("B)", side = 3, line = 1, adj = 0, font = 2, cex = 1.2, col = "grey30")
dev.off()



#### DUNG BEETLE ANALYSIS ####

## Dung Beetle Abundance ###
DBID <- read.table("Dung Beetle ID Data_JG.txt", header = TRUE)
View(DBID)

## Combine Trap types P & B (pitfall and bottle traps)  ##
DB_ID <- DBID %>%
  group_by(Site, Date, Round) %>%
  summarise(
    Sum = sum(Sum),
    Coprophanaeus_solisi = sum(Coprophanaeus_solisi),
    Coprophanaeus_corythus = sum(Coprophanaeus_corythus),
    Coprophanaeus_pecki = sum(Coprophanaeus_pecki),
    Onthophagus_praecellens = sum(Onthophagus_praecellens),
    Onthophagus_acuminatus = sum(Onthophagus_acuminatus),
    Onthophagus_coriaceoumbrosus = sum(Onthophagus_coriaceoumbrosus),
    Onthophagus_marginicollis = sum(Onthophagus_marginicollis),
    Onthophagus_sharpi = sum(Onthophagus_sharpi),
    Eurysternus_foedus = sum(Eurysternus_foedus),
    Eurysternus_plebejus = sum(Eurysternus_plebejus),
    Scybalocanthon_moniliatus = sum(Scybalocanthon_moniliatus),
    Deltochilium_pseudoparile = sum(Deltochilium_pseudoparile),
    Deltochilium_gibbosum = sum(Deltochilium_gibbosum),
    Copris_brevicornis = sum(Copris_brevicornis),
    Sylvicanthon_aequinoctialis = sum(Sylvicanthon_aequinoctialis),
    Canthon_mutabilis = sum(Canthon_mutabilis),
    Pedaridium_pilosum = sum(Pedaridium_pilosum),
    Canthidium_centrale = sum(Canthidium_centrale)
  )


# Add a Treatment column based on the Site name
DB_ID <- DB_ID %>%
  mutate(Treatment = ifelse(grepl("N", Site), "control", 
                            ifelse(grepl("E", Site), "exclosure", NA)))

# Create a tibble with just Site and Treatment columns to check if all is in order
site_treatment_tibble <- DB_ID %>%
  select(Site, Treatment)

# Add a Habitat column based on Site
DB_ID <- DB_ID %>%
  mutate(Habitat = ifelse(grepl("G", Site), "grassland", 
                          ifelse(grepl("F", Site), "forest", NA)))

DB_AB <- DB_ID[, c("Site", "Sum","Date", "Round", "Treatment", "Habitat")]

DB_AB <- DB_AB%>%
  group_by(Site, Round, Treatment, Habitat) %>%
  summarise(abundance = round(mean(Sum), 2))


# Add the 'Season' column based on the 'Round' values
DB_AB<- DB_AB %>%
  mutate(Season = ifelse(Round %in% 1:4, "wet", "dry"))


## Convert to factors

DB_AB$Treatment <- as.factor(DB_AB$Treatment)
summary(DB_AB$Treatment) # 2 treatments: Control & Exclosure. 16 observations each.
DB_AB$Habitat <- as.factor(DB_AB$Habitat)
summary(DB_AB$Habitat) # 2 habitats: Forest & Grassland. 16 observations each.
DB_AB$Round <- as.factor(DB_AB$Round)
summary(DB_AB$Round) # 8 rounds. 4 observations each.
DB_AB$Season <- as.factor(DB_AB$Season)
summary(DB_AB$Season) # 2 seasons: Wet & Dry. 16 observations each.


# Rename Site to pig_ID

DB_AB <- DB_AB %>%
  rename(pig_ID = Site)

lm <- lm(abundance ~ Treatment, data=DB_AB)
plot(lm)
hist(DB_AB$abundance) # right skewed.

# Plot abundance in treatments (control and exclosure) and check outliers.

boxplot(abundance ~ Treatment, data=DB_AB)$out #Outliers are not out of the ordinary


###### Table 3. Model selection ######

DB_AB$abun_integer <- round(DB_AB$abundance, 0)

lm  <- glmmTMB(abun_integer ~ Treatment + (1|Round), data=DB_AB, family=nbinom2)
lm1 <- glmmTMB(abun_integer ~ Habitat + (1|Round), data=DB_AB, family=nbinom2)
lm2 <- glmmTMB(abun_integer ~ Season + (1|Round), data=DB_AB, family=nbinom2)
lm3 <- glmmTMB(abun_integer ~ Treatment + Habitat + (1|Round), data=DB_AB, family=nbinom2)
lm4 <- glmmTMB(abun_integer ~ Treatment + Habitat + Season + (1|Round), data=DB_AB, family=nbinom2)
lm5 <- glmmTMB(abun_integer ~ Treatment * Habitat + (1|Round), data=DB_AB, family=nbinom2)
lm6 <- glmmTMB(abun_integer ~ Treatment * Habitat  + Season + (1|Round), data=DB_AB, family=nbinom2)
lm7 <- glmmTMB(abun_integer ~ Treatment + Season + (1|Round), data=DB_AB, family=nbinom2)
lm8 <- glmmTMB(abun_integer ~ Treatment * Season + (1|Round), data=DB_AB, family=nbinom2)
lm9 <- glmmTMB(abun_integer ~ Treatment * Season + Habitat + (1|Round), data=DB_AB, family=nbinom2)
lmNull <- glmmTMB(abun_integer ~ (1|Round), data=DB_AB, family=nbinom2)

model_table <- model.sel(lm, lm1, lm2, lm3, lm4, lm5,lm6,lm7,lm8,lm9, lmNull) #lm1 is best fit model, same as other DB abundance analysis
print(model_table)
r.squaredGLMM(lm1) # Habitat best explains dung beetle abundance
summary(lm1)
confint(lm1)


## Does DB abundance affect difference in carcass decomposition in the absence of vultures?

DB_ex <- DB_AB[DB_AB$Treatment=="exclosure",]
DB_ex <- left_join(DB_ex, Decomp[,c("pig_ID", "rate")])
View(DB_ex)

rlm1 <- lm(rate ~ abun_integer, data=DB_ex)
rlm2 <- lm(rate ~ 1 , data=DB_ex)
model_table <-model.sel(rlm1,rlm2) # rlm2 better fit, without DB abundance
print(model_table)
summary(rlm1)


###### Fig. 4. Plot DB abundance  #######
# Wet Season - Control
lmcon_wet <- lm(abun_integer ~ Habitat, data = subset(DB_AB, Treatment == "control" & Season == "wet"))
plotF1_wet <- effect_plot(
  lmcon_wet, pred = Habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1,
  main.title = "Control - Wet Season", x.label = "Habitat", y.label = "Dung Beetle Abundance (nr/day)",
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Dry Season - Control
lmcon_dry <- lm(abun_integer ~ Habitat, data = subset(DB_AB, Treatment == "control" & Season == "dry"))
plotF1_dry <- effect_plot(
  lmcon_dry, pred = Habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1,
  main.title = "Control - Dry Season", x.label = "Habitat", y.label = "Dung Beetle Abundance (nr/day)",
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Wet Season - Exclosure
lmexc_wet <- lm(abun_integer ~ Habitat, data = subset(DB_AB, Treatment == "exclosure" & Season == "wet"))
plotF2_wet <- effect_plot(
  lmexc_wet, pred = Habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1,
  main.title = "Exclosure - Wet Season", x.label = "Habitat", y.label = "Dung Beetle Abundance (nr/day)",
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Dry Season - Exclosure
lmexc_dry <- lm(abun_integer ~ Habitat, data = subset(DB_AB, Treatment == "exclosure" & Season == "dry"))
plotF2_dry <- effect_plot(
  lmexc_dry, pred = Habitat, interval = TRUE, plot.points = TRUE, jitter = 0.1,
  main.title = "Exclosure - Dry Season", x.label = "Habitat", y.label = "Dung Beetle Abundance (nr/day)",
  line.thickness = 0.7
) + 
  theme(plot.title.position = 'plot', plot.title = element_text(hjust = 0.7)) + ylim(0, 20)

# Combine Plots in a Grid
par(mfrow = c(1, 2))
plot_dung_season <- grid.arrange(plotF1_wet, plotF1_dry, plotF2_wet, plotF2_dry, nrow = 2, ncol = 2)

# Combined Graph for Both Treatments Across Seasons
plot_combined <- effect_plot(lm1, pred = "Habitat")


## Panel A

pdf("Figure4.pdf", height=6, width=10)

par(mfrow = c(1, 2))
plotF1 <- effect_plot(lm1, pred="Habitat")

# JULIA - HERE ARE THE PREDICTIONS
plotF1$data

plotCI(
  c(1.5, 2.9), c(plot_combined$data$abun_integer),
  ui = c(plot_combined$data$ymax),
  li = c(plot_combined$data$ymin),
  xlim = c(0.9, 3.5), las = 1,
  ylim = c(0, 20), pch = 19, sfrac = 0.00001,
  ylab = expression("Dung beetle abundance (nr day"^{-1}*")"),
  xlab = "", xaxt = "n", lwd = 2
)
abline(v = 2.2, lty = 2)

mtext("A)", side = 3, line = 1, adj = 0, font = 2, cex = 1.2, col = "grey30")
# Add Raw Data for Wet and Dry Seasons

# Forest control (Wet)
n_points <- sum(DB_AB$Treatment == "control" & DB_AB$Habitat == "forest" & DB_AB$Season == "wet")
points(jitter(rep(1.2, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "control" & DB_AB$Habitat == "forest" & DB_AB$Season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Forest control (Dry)
n_points <- sum(DB_AB$Treatment == "control" & DB_AB$Habitat == "forest" & DB_AB$Season == "dry")
points(jitter(rep(1.2, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "control" & DB_AB$Habitat == "forest" & DB_AB$Season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Forest exclosure (Wet)
n_points <- sum(DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "forest" & DB_AB$Season == "wet")
points(jitter(rep(1.8, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "forest" & DB_AB$Season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Forest exclosure (Dry)
n_points <- sum(DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "forest" & DB_AB$Season == "dry")
points(jitter(rep(1.8, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "forest" & DB_AB$Season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

# Grassland control (Wet)
n_points <- sum(DB_AB$Treatment == "control" & DB_AB$Habitat == "grassland" & DB_AB$Season == "wet")
points(jitter(rep(2.6, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "control" & DB_AB$Habitat == "grassland" & DB_AB$Season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Grassland control (Dry)
n_points <- sum(DB_AB$Treatment == "control" & DB_AB$Habitat == "grassland" & DB_AB$Season == "dry")
points(jitter(rep(2.6, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "control" & DB_AB$Habitat == "grassland" & DB_AB$Season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry


# Grassland exclosure (Wet)
n_points <- sum(DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "grassland" & DB_AB$Season == "wet")
points(jitter(rep(3.2, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "grassland" & DB_AB$Season == "wet"], 
       pch = 19, cex = 0.7, col = rgb(0, 0, 1, 0.4)) # Wet

# Grassland exclosure (Dry)
n_points <- sum(DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "grassland" & DB_AB$Season == "dry")
points(jitter(rep(3.2, time = n_points), 1), 
       DB_AB$abun_integer[DB_AB$Treatment == "exclosure" & DB_AB$Habitat == "grassland" & DB_AB$Season == "dry"], 
       pch = 19, cex = 0.7, col = rgb(1, 0, 0, 0.4)) # Dry

axis(1, at = c(1.2, 1.8, 2.6, 3.2),
     labels = c("Control", "Exclusion", "Control", "Exclusion"))

mtext(expression(bold("Forest")), las = 1, side = 1, line = 2.4, outer = FALSE, at = c(1.5))
mtext(expression(bold("Grassland")), las = 1, side = 1, line = 2.4, outer = FALSE, at = c(2.9))

lines(c(1.5, 2.9),
      c(plot_combined$data$abun_integer), lty = 2)


## Panel B

plot1 <- effect_plot(rlm1, pred=abun_integer, interval=TRUE, robust=TRUE, plot.points=TRUE, jitter=.1, main.title="Control", x.label="Habitat", y.label="Decomposition rate (kg lost/day)", line.thickness=0.7) +
  theme(plot.title.position='plot', plot.title=element_text(hjust=0.7)) + ylim(0, 20)

# JULIA - HERE ARE THE PREDICTIONS
plot1$data


plot(DB_ex$rate ~ DB_ex$abun_integer, ylim=c(0, 13), las=1,
     ylab=expression("Carcass decomposition rate (kg lost day"^{-1}*")"),
     xlab=expression("Dung beetle abundance (nr day"^{-1}*")"), type="n")

polygon(c(plot1$data$abun_integer, rev(plot1$data$abun_integer)),
        c(plot1$data$ymax, rev(plot1$data$ymin)),
        lwd=2, border=rgb(0.9, 0.9, 0.9), col=rgb(0.9, 0.9, 0.9))
lines(plot1$data$rate ~ plot1$data$abun_integer, lwd=2)
points(DB_ex$rate ~ DB_ex$abun_integer, pch=19,
       cex=0.7, col=rgb(0, 0, 0, 0.4))
mtext("B)", side = 3, line = 1, adj = 0, font = 2, cex = 1.2, col = "grey30")

dev.off()


###### Appendix DB species table #####

DBID <- read.table("Dung Beetle ID Data.txt", header = TRUE)
View(DBID)

## Combine Trap types P & B (pitfall and bottle traps)  ##
DB_ID <- DBID %>%
  group_by(Site, Date, Round) %>%
  summarise(
    Sum = sum(Sum),
    Coprophanaeus_solisi = sum(Coprophanaeus_solisi),
    Coprophanaeus_corythus = sum(Coprophanaeus_corythus),
    Coprophanaeus_pecki = sum(Coprophanaeus_pecki),
    Onthophagus_praecellens = sum(Onthophagus_praecellens),
    Onthophagus_acuminatus = sum(Onthophagus_acuminatus),
    Onthophagus_coriaceoumbrosus = sum(Onthophagus_coriaceoumbrosus),
    Onthophagus_marginicollis = sum(Onthophagus_marginicollis),
    Onthophagus_sharpi = sum(Onthophagus_sharpi),
    Eurysternus_foedus = sum(Eurysternus_foedus),
    Eurysternus_plebejus = sum(Eurysternus_plebejus),
    Scybalocanthon_moniliatus = sum(Scybalocanthon_moniliatus),
    Deltochilium_pseudoparile = sum(Deltochilium_pseudoparile),
    Deltochilium_gibbosum = sum(Deltochilium_gibbosum),
    Copris_brevicornis = sum(Copris_brevicornis),
    Sylvicanthon_aequinoctialis = sum(Sylvicanthon_aequinoctialis),
    Canthon_mutabilis = sum(Canthon_mutabilis),
    Pedaridium_pilosum = sum(Pedaridium_pilosum),
    Canthidium_centrale = sum(Canthidium_centrale)
  )


# Add a Treatment column based on the Site name
DB_ID <- DB_ID %>%
  mutate(Treatment = ifelse(grepl("N", Site), "control", 
                            ifelse(grepl("E", Site), "exclosure", NA)))

# Create a tibble with just Site and Treatment columns to check if all is in order
site_treatment_tibble <- DB_ID %>%
  select(Site, Treatment)

# Add a Habitat column based on Site
DB_ID <- DB_ID %>%
  mutate(Habitat = ifelse(grepl("G", Site), "grassland", 
                          ifelse(grepl("F", Site), "forest", NA)))

# Reshape the dataset into long format for species counts
species_table <- DB_ID%>%
  pivot_longer(
    cols = c(
      "Coprophanaeus_solisi", "Coprophanaeus_corythus", "Coprophanaeus_pecki",
      "Onthophagus_praecellens", "Onthophagus_acuminatus", "Onthophagus_coriaceoumbrosus",
      "Onthophagus_marginicollis", "Onthophagus_sharpi", "Eurysternus_foedus",
      "Eurysternus_plebejus", "Scybalocanthon_moniliatus", "Deltochilium_pseudoparile",
      "Deltochilium_gibbosum", "Copris_brevicornis", "Sylvicanthon_aequinoctialis",
      "Canthon_mutabilis", "Pedaridium_pilosum", "Canthidium_centrale"
    ),
    names_to = "Species",
    values_to = "Count"
  )

View(species_table)

# Create Exclosure and Control columns based on Treatment
DB_SP <- species_table %>%
  mutate(
    Exclosure = if_else(Treatment == "exclosure", Count, 0),
    Control = if_else(Treatment == "control", Count, 0),
    Forest = if_else(Habitat == "forest", Count, 0),
    Grassland = if_else(Habitat == "grassland", Count, 0)
  )

#Group by species name and sum up counts for Exclosure, Control, Forest, and Grassland
final_species_table <- DB_SP %>%
  group_by(Species) %>%
  summarise(
    Count = sum(Count),            # Total count for the species
    Exclosure = sum(Exclosure),    # Sum of Exclosure counts
    Control = sum(Control),        # Sum of Control counts
    Forest = sum(Forest),          # Sum of Forest counts
    Grassland = sum(Grassland)     # Sum of Grassland counts
  )

# Rename the Count column to Total
names(final_species_table)[names(final_species_table) == "Count"] <- "Total"
View(final_species_table)

# Export to a CSV file
write.csv(final_species_table, "DB_Species_Table.csv", row.names = FALSE)


